import os
import json
import time
import logging
import sys
import configparser
import random
import hashlib
import hmac
import base64
from datetime import datetime
from cryptography.hazmat.primitives.ciphers import Cipher, algorithms, modes
from cryptography.hazmat.backends import default_backend
from telethon.sync import TelegramClient
from telethon.errors import UserPrivacyRestrictedError, FloodWaitError, PhoneCodeInvalidError, PhoneNumberBannedError
from telethon.tl.functions.messages import AddChatUserRequest, ImportChatInviteRequest
from telethon.tl.functions.channels import InviteToChannelRequest
from telethon.tl.functions.contacts import GetContactsRequest
from telethon.tl.types import Channel, Chat, User
from tqdm import tqdm
from colorama import init, Fore, Style

init(autoreset=True)

# ======================== SISTEM LISENSI TERENKRIPSI ========================
LICENSE_FILE = "license.key"
HMAC_SECRET = b"rahasia_super_aman_123"
ENCRYPTION_KEY = hashlib.sha256(b"enkripsi_super_kuat_456").digest()

def get_hwid():
    import uuid
    return hashlib.sha256(str(uuid.getnode()).encode()).hexdigest()

def decrypt_license():
    if not os.path.exists(LICENSE_FILE):
        hwid = get_hwid()
        print("❌ Lisensi tidak ditemukan.")
        print(f"📌 Kirim HWID ini ke developer untuk aktivasi: {hwid}")
        sys.exit(1)

    try:
        with open(LICENSE_FILE, "r") as f:
            content = f.read().strip()

        blob = base64.b64decode(content)
        iv = blob[:16]
        ciphertext = blob[16:]

        cipher = Cipher(algorithms.AES(ENCRYPTION_KEY), modes.CBC(iv), backend=default_backend())
        decryptor = cipher.decryptor()
        padded = decryptor.update(ciphertext) + decryptor.finalize()

        pad_len = padded[-1]
        decrypted = padded[:-pad_len].decode()
        return decrypted
    except Exception as e:
        print(f"❌ Gagal mendekripsi lisensi: {e}")
        sys.exit(1)

def check_license():
    decrypted = decrypt_license()
    parts = decrypted.split('|')
    if len(parts) != 3:
        print("❌ Format lisensi tidak valid.")
        sys.exit(1)

    hwid, expire_date, provided_sig = parts
    expected_sig = hmac.new(HMAC_SECRET, f"{hwid}|{expire_date}".encode(), hashlib.sha256).hexdigest()

    if provided_sig != expected_sig:
        print("❌ Tanda tangan tidak valid.")
        sys.exit(1)

    if hwid != get_hwid():
        print("❌ Lisensi tidak cocok dengan perangkat ini.")
        sys.exit(1)

    try:
        if datetime.now() > datetime.strptime(expire_date, "%Y-%m-%d"):
            print("❌ Lisensi telah kedaluwarsa.")
            sys.exit(1)
    except:
        print("❌ Format tanggal salah.")
        sys.exit(1)

# ======================== SCRIPT OTOMASI TELEGRAM ========================
SESSIONS_FOLDER = 'sessions'
PROGRESS_FILE_TEMPLATE = 'progress_{phone_number}.json'
PHONE_FILE = 'phone.txt'
summary_log = []

def setup_logging():
    if hasattr(sys.stdout, 'reconfigure'):
        sys.stdout.reconfigure(encoding='utf-8')
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(levelname)s - %(message)s',
        handlers=[
            logging.StreamHandler(sys.stdout),
            logging.FileHandler("telethon_automation.log", encoding="utf-8")
        ]
    )

setup_logging()

def parse_random_range(value, key_name):
    try:
        value = value.replace(' ', '')
        if '-' in value:
            min_val, max_val = map(int, value.split('-'))
            return random.randint(min_val, max_val)
        return int(value)
    except Exception:
        raise ValueError(f"Invalid range format for {key_name}. Use 'min-max' or a single integer.")

def load_config():
    config = configparser.ConfigParser()
    config.read('config.ini')
    if 'Telegram' not in config:
        raise ValueError("Missing 'Telegram' section in config.ini")

    telegram_config = config['Telegram']

    return {
        'api_id': telegram_config.getint('api_id'),
        'api_hash': telegram_config.get('api_hash'),
        'target_group_id': telegram_config.getint('target_group_id'),
        'target_group_invite_link': telegram_config.get('target_group_invite_link'),
        'allow_channel_processing': telegram_config.getboolean('allow_channel_processing'),
        'pause_after_batch': parse_random_range(telegram_config.get('pause_after_batch'), 'pause_after_batch'),
        'batch_sleep_time': parse_random_range(telegram_config.get('batch_sleep_time'), 'batch_sleep_time'),
        'max_users_per_account': telegram_config.getint('max_users_per_account'),
        'contacts_per_cycle': telegram_config.getint('contacts_per_cycle'),
        'account_sleep_time': parse_random_range(telegram_config.get('account_sleep_time'), 'account_sleep_time'),
        'max_sleep_time': parse_random_range(telegram_config.get('max_sleep_time'), 'max_sleep_time'),
        'delay_between_adds': telegram_config.get('delay_between_adds')
    }

def load_phone_numbers():
    if not os.path.exists(PHONE_FILE):
        raise FileNotFoundError(f"Phone file '{PHONE_FILE}' not found.")
    with open(PHONE_FILE, 'r') as f:
        return [line.strip() for line in f if line.strip()]

def load_progress(phone_number):
    path = PROGRESS_FILE_TEMPLATE.format(phone_number=phone_number)
    if os.path.exists(path):
        try:
            return set(json.load(open(path)))
        except Exception:
            logging.warning(Fore.YELLOW + f"Progress file corrupt for {phone_number}, resetting.")
    return set()

def save_progress(phone_number, processed_ids):
    path = PROGRESS_FILE_TEMPLATE.format(phone_number=phone_number)
    with open(path, 'w', encoding='utf-8') as f:
        json.dump(list(processed_ids), f)

def auto_join_group(client, invite_link):
    try:
        logging.info(Fore.CYAN + f"Trying to join group: {invite_link}")
        hash_part = invite_link.split('+')[-1]
        client(ImportChatInviteRequest(hash_part))
        logging.info(Fore.CYAN + "Successfully joined group.")
        return True
    except Exception as e:
        if "already a participant" in str(e):
            logging.info(Fore.YELLOW + "Already a member of the group, skipping join.")
            return True
        logging.error(Fore.RED + f"Join failed: {e}")
        return False

def check_account_status(client):
    try:
        client.get_me()
        return True
    except PhoneNumberBannedError:
        logging.error(Fore.RED + "Banned number.")
    except PhoneCodeInvalidError:
        logging.error(Fore.RED + "Invalid code.")
    except Exception as e:
        logging.error(Fore.RED + f"Account error: {e}")
    return False

def remove_banned_number(phone_number, phone_numbers):
    phone_numbers.remove(phone_number)
    with open(PHONE_FILE, 'w') as f:
        for number in phone_numbers:
            f.write(f"{number}\n")
    logging.info(Fore.MAGENTA + f"Removed banned number: {phone_number}")

def check_group_type(dialog):
    entity = dialog.entity
    if isinstance(entity, Channel):
        return "Channel" if entity.broadcast else "Supergroup"
    elif isinstance(entity, Chat):
        return "Group"
    return "Unknown"

def add_contact(client, group, contact, group_type, processed_ids):
    try:
        if group_type == "Supergroup":
            client(InviteToChannelRequest(channel=group.entity, users=[contact.id]))
        else:
            client(AddChatUserRequest(chat_id=group.id, user_id=contact.id, fwd_limit=0))
        logging.info(Fore.GREEN + f"Added {contact.first_name} ({contact.id})")
        processed_ids.add(contact.id)
        return True
    except UserPrivacyRestrictedError:
        logging.warning(Fore.YELLOW + f"Privacy restriction for {contact.first_name} ({contact.id})")
    except FloodWaitError as e:
        logging.warning(Fore.YELLOW + f"Flood wait for {e.seconds} seconds. Retrying.")
        time.sleep(e.seconds)
    except Exception as e:
        logging.error(Fore.RED + f"Failed to add {contact.first_name} ({contact.id}): {e}")
    return False

def process_contacts_for_phone(client, phone_number, config, contacts_to_add):
    target_group_id = config['target_group_id']
    invite_link = config['target_group_invite_link']
    allow_channel_processing = config['allow_channel_processing']
    max_users_per_account = config['max_users_per_account']
    delay_range = config['delay_between_adds']

    processed_ids = load_progress(phone_number)
    try:
        auto_join_group(client, invite_link)

        group = next((d for d in client.iter_dialogs() if d.id == target_group_id), None)
        if not group:
            logging.warning(Fore.RED + f"Group ID '{target_group_id}' not found in dialogs.")
            return 0

        group_type = check_group_type(group)
        if group_type == "Channel" and not allow_channel_processing:
            return 0

        contacts = client(GetContactsRequest(hash=0)).users
        to_process = [c for c in contacts if c.id not in processed_ids]
        added = 0

        progress_bar = tqdm(to_process[:max_users_per_account],
                            desc=Fore.CYAN + f"[{phone_number}] Adding", unit="user")

        for contact in progress_bar:
            if added >= contacts_to_add:
                break
            if add_contact(client, group, contact, group_type, processed_ids):
                added += 1
                save_progress(phone_number, processed_ids)

                if isinstance(delay_range, str) and '-' in delay_range:
                    min_d, max_d = map(int, delay_range.split('-'))
                    actual_delay = random.randint(min_d, max_d)
                else:
                    actual_delay = int(delay_range)

                logging.info(Fore.BLUE + f"⏳ Waiting {actual_delay}s before next.")
                time.sleep(actual_delay)

        msg = f"[{phone_number}] ✅ Added {added} contact(s)." if added else f"[{phone_number}] ⚠️ No contact added."
        print(Fore.GREEN + Style.BRIGHT + msg)
        summary_log.append(msg)
        return added
    finally:
        client.disconnect()

def process_all_accounts(phone_numbers, config):
    api_id = config['api_id']
    api_hash = config['api_hash']
    pause_after_batch = config['pause_after_batch']
    account_sleep_time = config['account_sleep_time']
    max_sleep_time = config['max_sleep_time']
    batch_sleep_time = config['batch_sleep_time']
    contacts_per_cycle = config['contacts_per_cycle']

    os.makedirs(SESSIONS_FOLDER, exist_ok=True)

    while True:
        finished = True
        for phone_number in phone_numbers[:]:
            session_path = os.path.join(SESSIONS_FOLDER, f"{phone_number}.session")
            client = TelegramClient(session_path, api_id, api_hash)
            client.start(phone=phone_number)

            if not check_account_status(client):
                remove_banned_number(phone_number, phone_numbers)
                continue

            try:
                added = process_contacts_for_phone(client, phone_number, config, contacts_per_cycle)
                finished &= (added == 0)
            except FloodWaitError as e:
                time.sleep(e.seconds)
            except Exception as e:
                logging.error(Fore.RED + f"Unexpected error: {e}")
            finally:
                client.disconnect()

            sleep_time = random.randint(account_sleep_time, max_sleep_time)
            logging.info(Fore.CYAN + f"Sleeping {sleep_time}s.")
            time.sleep(sleep_time)

        if finished:
            break

        logging.info(Fore.CYAN + f"Pausing batch: {batch_sleep_time}s")
        time.sleep(batch_sleep_time)
        logging.info(Fore.CYAN + f"Extra pause: {pause_after_batch}s")
        time.sleep(pause_after_batch)

    with open("summary_report.txt", "w", encoding="utf-8") as f:
        f.write("Telegram Automation Summary\n===========================\n")
        f.write("\n".join(summary_log))
    logging.info(Fore.CYAN + "📄 Summary saved to: summary_report.txt")

def main():
    check_license()
    config = load_config()
    phone_numbers = load_phone_numbers()
    process_all_accounts(phone_numbers, config)

if __name__ == "__main__":
    main()
